<?php
declare(strict_types=1);
require_once __DIR__ . '/includes/config.php';
require_once __DIR__ . '/includes/auth.php';

ensure_session();

$success = '';
$error = '';

/*
  Optional:
  Block access if an admin already exists
*/
$adminExists = db()->query("
    SELECT COUNT(*) c
    FROM user_roles ur
    JOIN roles r ON r.id = ur.role_id
    WHERE r.role_key = 'admin'
")->fetch()['c'] ?? 0;

if ((int)$adminExists > 0) {
    $error = "Administrator already exists. This page is disabled.";
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && !$error) {

    csrf_validate($_POST['_csrf'] ?? null);

    $regKey = trim((string)($_POST['registration_key'] ?? ''));
    $fullName = trim((string)($_POST['full_name'] ?? ''));
    $email = trim((string)($_POST['email'] ?? ''));
    $password = (string)($_POST['password'] ?? '');
    $confirm = (string)($_POST['confirm_password'] ?? '');

    if ($regKey !== ADMIN_REGISTRATION_KEY) {
        $error = "Invalid registration key.";
    }
    elseif ($fullName === '' || $email === '' || $password === '') {
        $error = "All fields are required.";
    }
    elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = "Invalid email format.";
    }
    elseif (strlen($password) < 8) {
        $error = "Password must be at least 8 characters.";
    }
    elseif ($password !== $confirm) {
        $error = "Passwords do not match.";
    }
    else {

        // Check duplicate email
        $stmt = db()->prepare("SELECT id FROM users WHERE email = ? LIMIT 1");
        $stmt->execute([$email]);
        if ($stmt->fetch()) {
            $error = "Email already exists.";
        }
        else {

            try {
                db()->beginTransaction();

                $hash = password_hash($password, PASSWORD_DEFAULT);

                $stmt = db()->prepare("
                    INSERT INTO users (full_name, email, password_hash, status)
                    VALUES (?, ?, ?, 'active')
                ");
                $stmt->execute([$fullName, $email, $hash]);

                $userId = (int)db()->lastInsertId();

                // Assign admin role
                $roleId = db()->query("SELECT id FROM roles WHERE role_key='admin' LIMIT 1")->fetch()['id'];

                $stmt = db()->prepare("INSERT INTO user_roles (user_id, role_id) VALUES (?, ?)");
                $stmt->execute([$userId, $roleId]);

                // Audit log
                db()->prepare("
                    INSERT INTO audit_logs (actor_user_id, action, entity_type, entity_id, ip_address, user_agent)
                    VALUES (?, 'admin_created', 'users', ?, ?, ?)
                ")->execute([
                    $userId,
                    $userId,
                    $_SERVER['REMOTE_ADDR'] ?? '',
                    $_SERVER['HTTP_USER_AGENT'] ?? ''
                ]);

                db()->commit();

                $success = "Administrator account successfully created. You may now login.";
            }
            catch (Throwable $e) {
                db()->rollBack();
                $error = "Registration failed. Please try again.";
            }
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title>SecureHRMS | Admin Registration</title>

<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/admin-lte@3.2/dist/css/adminlte.min.css">
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/@fortawesome/fontawesome-free@5.15.4/css/all.min.css">

<style>
body {
    background: linear-gradient(135deg,#0B0F19,#1a1f2e);
}
.register-box {
    margin-top: 5%;
}
.card-primary.card-outline {
    border-top: 3px solid #C9A24D;
}
.btn-gold {
    background:#C9A24D;
    border:none;
    color:#000;
    font-weight:600;
}
.btn-gold:hover {
    background:#d8b25a;
}
</style>
</head>
<body class="hold-transition register-page">

<div class="register-box">
  <div class="card card-outline card-primary">
    <div class="card-header text-center">
      <b>SecureHRMS</b> | Admin Setup
    </div>
    <div class="card-body">

      <p class="login-box-msg">Create Initial Administrator Account</p>

      <?php if ($error): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
      <?php endif; ?>

      <?php if ($success): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
        <div class="text-center mt-3">
          <a href="login.php" class="btn btn-gold">
            <i class="fas fa-sign-in-alt"></i> Go to Login
          </a>
        </div>
      <?php elseif (!$error): ?>

      <form method="post">
        <input type="hidden" name="_csrf" value="<?= htmlspecialchars(csrf_token()) ?>">

        <div class="input-group mb-3">
          <input type="text" name="registration_key" class="form-control" placeholder="Admin Registration Key" required>
          <div class="input-group-append">
            <div class="input-group-text"><span class="fas fa-key"></span></div>
          </div>
        </div>

        <div class="input-group mb-3">
          <input type="text" name="full_name" class="form-control" placeholder="Full Name" required>
          <div class="input-group-append">
            <div class="input-group-text"><span class="fas fa-user"></span></div>
          </div>
        </div>

        <div class="input-group mb-3">
          <input type="email" name="email" class="form-control" placeholder="Email" required>
          <div class="input-group-append">
            <div class="input-group-text"><span class="fas fa-envelope"></span></div>
          </div>
        </div>

        <div class="input-group mb-3">
          <input type="password" name="password" class="form-control" placeholder="Password (min 8 chars)" required>
          <div class="input-group-append">
            <div class="input-group-text"><span class="fas fa-lock"></span></div>
          </div>
        </div>

        <div class="input-group mb-3">
          <input type="password" name="confirm_password" class="form-control" placeholder="Confirm Password" required>
          <div class="input-group-append">
            <div class="input-group-text"><span class="fas fa-lock"></span></div>
          </div>
        </div>

        <button type="submit" class="btn btn-gold btn-block">
          <i class="fas fa-user-shield"></i> Create Admin
        </button>

      </form>

      <?php endif; ?>

    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/plugins/jquery/jquery.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/admin-lte@3.2/plugins/bootstrap/js/bootstrap.bundle.min.js"></script>

</body>
</html>