<?php
declare(strict_types=1);
require_once __DIR__ . '/config.php';

function ensure_session(): void {
    if (session_status() !== PHP_SESSION_ACTIVE) session_start();
}

function is_logged_in(): bool {
    ensure_session();
    return !empty($_SESSION['user']['id']);
}

function require_login(): void {
    if (!is_logged_in()) {
        header('Location: ' . BASE_URL . '/login.php');
        exit;
    }
}

function current_user(): array {
    ensure_session();
    return $_SESSION['user'] ?? [];
}

function load_user_roles(int $userId): array {
    $stmt = db()->prepare("
        SELECT r.role_key
        FROM user_roles ur
        JOIN roles r ON r.id = ur.role_id
        WHERE ur.user_id = ?
    ");
    $stmt->execute([$userId]);
    return array_column($stmt->fetchAll(), 'role_key');
}

function user_has_role(string $roleKey): bool {
    if (!is_logged_in()) return false;
    $roles = $_SESSION['user']['roles'] ?? [];
    return in_array($roleKey, $roles, true);
}

function require_role(array $roleKeys): void {
    require_login();
    foreach ($roleKeys as $rk) {
        if (user_has_role($rk)) return;
    }
    http_response_code(403);
    exit('Forbidden');
}