<?php
require_once __DIR__ . '/../config/auth.php';
require_once __DIR__ . '/../../config/database.php';

session_start();

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    die('Invalid request.');
}

if (
    empty($_POST['booking_id']) ||
    empty($_POST['status'])
) {
    die('Missing required fields.');
}

$allowed_statuses = [
    'pending',
    'assigned',
    'accepted',
    'on_the_way',
    'arrived',
    'started',
    'completed',
    'cancelled'
];

$new_status = $_POST['status'];

if (!in_array($new_status, $allowed_statuses)) {
    die('Invalid status.');
}

$db = Database::connect();

try {

    $db->beginTransaction();

    /*
    |--------------------------------------------------------------------------
    | Lock Booking Row (Prevents Race Conditions)
    |--------------------------------------------------------------------------
    */
    $stmt = $db->prepare("
        SELECT id, status
        FROM bookings
        WHERE id = ?
        FOR UPDATE
    ");
    $stmt->execute([$_POST['booking_id']]);
    $booking = $stmt->fetch();

    if (!$booking) {
        throw new Exception('Booking not found.');
    }

    $current_status = $booking['status'];

    /*
    |--------------------------------------------------------------------------
    | Prevent Duplicate Update
    |--------------------------------------------------------------------------
    */
    if ($current_status === $new_status) {
        throw new Exception('Booking already has this status.');
    }

    /*
    |--------------------------------------------------------------------------
    | Optional Status Flow Control
    | (Prevents jumping backwards improperly)
    |--------------------------------------------------------------------------
    */
    $status_flow = [
        'pending'      => 1,
        'assigned'     => 2,
        'accepted'     => 3,
        'on_the_way'   => 4,
        'arrived'      => 5,
        'started'      => 6,
        'completed'    => 7,
        'cancelled'    => 8
    ];

    if (
        $new_status !== 'cancelled' &&
        $status_flow[$new_status] < $status_flow[$current_status]
    ) {
        throw new Exception('Invalid status transition.');
    }

    /*
    |--------------------------------------------------------------------------
    | Update Booking Status
    |--------------------------------------------------------------------------
    */
    $stmt = $db->prepare("
        UPDATE bookings
        SET status = ?
        WHERE id = ?
    ");
    $stmt->execute([$new_status, $booking['id']]);

    /*
    |--------------------------------------------------------------------------
    | Insert Timeline Entry
    |--------------------------------------------------------------------------
    */
    $stmt = $db->prepare("
        INSERT INTO booking_timeline
        (booking_id, status, updated_by)
        VALUES (?, ?, 'admin')
    ");
    $stmt->execute([
        $booking['id'],
        $new_status
    ]);

    $db->commit();

    header("Location: view.php?id=" . $booking['id']);
    exit;

} catch (Exception $e) {

    if ($db->inTransaction()) {
        $db->rollBack();
    }

    die($e->getMessage());
}
